/*
 * $Id$
 *
 * Create FTEX files.
 *
 * version 0.1
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <FreeImage.h>
#include <string.h>
#include <ATI_Compress.h>
#include "err.h"
#include "ftex.h"
#include "hexdump.h"
#include "getopt.h"
#include "util.h"

void usage(void)
{
    printf("cftex: [options] <infile> <outfile>\n"
            "Options:\n"
            "-r <rescale type>      Use <rescale type> when generating mipmaps,\n"
            "                       where <rescale type> is one of:\n"
            "                       box, bilinear, bspline, bicubic, catmullrom,\n"
            "                       or lancsoz3. Defaults to bspline.\n"
            "-q                     quiet mode - only errors are printed\n"
            "\n"
            "<outfile> should not have a file extention, both .ftu and .ftc files\n"
            "will be created.\n"
        );
}

int quiet;


struct filter_type {
  char *name;
  int type;
};

struct filter_type filters[] = {
  {"box", FILTER_BOX},
  {"bilinear", FILTER_BILINEAR},
  {"bspline", FILTER_BSPLINE},
  {"bicubic", FILTER_BICUBIC},
  {"catmullrom", FILTER_CATMULLROM},
  {"lancsoz3", FILTER_LANCZOS3},
  {NULL, -1}
};

bool is_power_of_2(unsigned int num)
{
  int count=0, i;

  for (i = 0; i < 32 ; i++)
  {
    if ((num >> i) & 1)
      count++;
    if (count > 1)
      return 0;
  }

  if (count == 1)
    return 1;
  else
    return 0;
}

int main(int argc, char *argv[])
{
  FIBITMAP *dib = NULL, *tmp, *mip;
  FREE_IMAGE_FORMAT fif = FIF_UNKNOWN;
  unsigned int mipmaps, w, h, bpp, i, j, mw, mh, size, sw, i2;
  ft_header fth;
  ft_dir ftd;
  FILE *u_fh, *c_fh;
  BYTE *data;
  BYTE t;
  char ch;
  char *progname, *fname;
  ATI_TC_Texture src, dst;
  ATI_TC_ERROR ati_err;
  int filter = FILTER_BSPLINE;

  progname = argv[0];
  quiet = 0;

  while ((ch = (char)getopt(argc, argv, "h?qr:")) != -1)
  {
      switch (ch) {
        case 'r':
          {
            struct filter_type *f;
            bool foundit;
            foundit = 0;

            i = 0;
            while (filters[i].name != NULL)
            {
              f = &filters[i];
              if (f->name == NULL)
                break;
              if (strcmp(f->name, optarg) == 0)
              {
                filter = f->type;
                foundit = 1;
              }
              i++;
            }
            if (!foundit)
            {
              errx(EXIT_FAILURE, "%s: invalid filter type: %s", progname, optarg);
            }
          }
          break;
        case 'q':
          quiet = 1;
          break;
        case '?':
        case 'h':
        default:
          usage();
          return EXIT_FAILURE;
      }
  }

  argc -= optind;
  argv += optind;

  if (argc != 2)
  {
    usage();
    return EXIT_FAILURE;
  }

  check_fi_version();

  if (!quiet)
    printf("Built using Freeimage version: %s\n", FreeImage_GetVersion());

  FreeImage_SetOutputMessage(FreeImageErrorHandler);

  if (!quiet)
  {
    i = 0;
    while (filters[i].name != NULL)
    {
      if (filters[i].type == filter)
      {
        printf("using %s for resampling the mipmaps\n", filters[i].name);
        break;
      }
      i++;
    }
  }

  fif = FreeImage_GetFileType(argv[0], 0);
  if(fif == FIF_UNKNOWN) {
    fif = FreeImage_GetFIFFromFilename(argv[0]);
  }
  if((fif != FIF_UNKNOWN) && FreeImage_FIFSupportsReading(fif)) {
    dib = FreeImage_Load(fif, argv[0], 0);
  }

  if (dib == NULL)
  {
    errx(EXIT_FAILURE, "%s: unable to load image %s", progname, argv[0]);
  }


  w = FreeImage_GetWidth(dib);
  h = FreeImage_GetHeight(dib);
  bpp = FreeImage_GetBPP(dib);

  if (bpp != 24)
  {
    tmp = FreeImage_ConvertTo24Bits(dib);
    FreeImage_Unload(dib);
    dib = tmp;
    bpp = FreeImage_GetBPP(dib);
  }
  if (FreeImage_GetColorType(dib) != FIC_RGB)
  {
    errx(EXIT_FAILURE, "%s: %s: Image must be RGB (I tried to convert it, but didn't succed(?))",
          progname, argv[0]);
  }

  if (!is_power_of_2(w))
  {
    errx(EXIT_FAILURE, "%s: %s: Image width must be a power of 2 (was %d)",
          progname, argv[0], w);
  }
  if (!is_power_of_2(h))
  {
    errx(EXIT_FAILURE, "%s: %s: Image height must be a power of 2 (was %d)",
          progname, argv[0], h);
  }

  if (FreeImage_IsTransparent(dib))
  {
    errx(EXIT_FAILURE, "%s: %s: Transparent images not supported",
          progname, argv[0]);
  }

  if (!quiet)
    printf("%s: Loaded %s: %dx%d %dbpp\n", progname, argv[0], w, h, bpp);

  fname = malloc(strlen(argv[1]) + 6);

  sprintf(fname, "%s.ftu", argv[1]);
  if ((u_fh = fopen(fname, "wb")) == NULL)
  {
    free(fname);
    errx(EXIT_FAILURE, "%s: unable to open output file",
          progname);
  }

  sprintf(fname, "%s.ftc", argv[1]);
  if ((c_fh = fopen(fname, "wb")) == NULL)
  {
    free(fname);
    errx(EXIT_FAILURE, "%s: unable to open output file",
          progname);
  }
  free(fname);

  mipmaps = 0;
  i = 1;
  while (((w < h ? w : h) / i) != 1) {
    i = i * 2;
    mipmaps++;
  }

  if (!quiet)
    printf("%s: generating %d mipmaps\n", progname, mipmaps);

  fth.height = h;
  fth.width = w;
  fth.mipmap_count = mipmaps + 1;
  fth.format_count = 1;
  fth.version = 1;
  fth.magic = FTEX_MAGIC;

  ftd.format = FTEX_FORMAT_RGB;
  ftd.where = sizeof(fth) + sizeof(ftd);

  fwrite(&fth, sizeof(fth), 1, u_fh);
  fwrite(&ftd, sizeof(ftd), 1, u_fh);

  ftd.format = FTEX_FORMAT_DXT1;
  fwrite(&fth, sizeof(fth), 1, c_fh);
  fwrite(&ftd, sizeof(ftd), 1, c_fh);

  /* first FTU image */
  size = w * h * 3;
  fwrite(&size, sizeof(size), 1, u_fh);

  sw = FreeImage_GetPitch(dib);
  data = (BYTE*)malloc(h * sw);

  FreeImage_ConvertToRawBits(data, dib, sw, 24, FI_RGBA_RED_MASK, FI_RGBA_GREEN_MASK, FI_RGBA_BLUE_MASK, FALSE);

  /* XXX FTU raw data is BGR??? , I need to be really sure i'm not making
   * a mistake somewhere.
   */
  for (i = 0 ; i < w * h * 3 ; i += 3)
  {
      t = data[i];
      data[i] = data[i + 2];
      data[i + 2] = t;
  }

  fwrite(data, size, 1, u_fh);
  free(data);
  /* end ftu */

  /* start first compressed image */
  tmp = FreeImage_ConvertTo32Bits(dib);
  sw = FreeImage_GetPitch(tmp);
  data = malloc(h * sw);

  FreeImage_ConvertToRawBits(data, tmp, sw, 32, FI_RGBA_RED_MASK, FI_RGBA_GREEN_MASK, FI_RGBA_BLUE_MASK, FALSE);

  src.dwSize = sizeof(src);
  src.dwWidth = w;
  src.dwHeight = h;
  src.dwPitch = sw;
  src.format = ATI_TC_FORMAT_ARGB_8888;
  src.dwDataSize = h * sw;
  src.pData = data;

  dst.dwSize = sizeof(dst);
  dst.dwWidth = w;
  dst.dwHeight = h;
  dst.dwPitch = 0;
  dst.format = ATI_TC_FORMAT_DXT1;
  dst.dwDataSize = ATI_TC_CalculateBufferSize(&dst);
  dst.pData = (ATI_TC_BYTE*) malloc(dst.dwDataSize);

//  printf("First Raw:\n");
//  pretty_print_block(src.pData, src.dwDataSize > 32 ? 32 : src.dwDataSize);

  if (!quiet)
    printf("Compressed texture size is %d : %d\n", dst.dwDataSize, ((w/4) * (h/4)) * 8);

  i = 0;
  ati_err = ATI_TC_ConvertTexture(&src, &dst, NULL, feedback, &i, NULL);

  if (!quiet)
    printf("\n");

  if (ati_err != ATI_TC_OK)
  {
    errx(EXIT_FAILURE, "%s: error while trying to compress texture: %d",
          progname, ati_err);
  }

//  printf("Compressed:\n");
//  pretty_print_block(dst.pData, dst.dwDataSize > 32 ? 32 : dst.dwDataSize);

  size = dst.dwDataSize;
  fwrite(&size, sizeof(size), 1, c_fh);
  fwrite(dst.pData, size, 1, c_fh);

  FreeImage_Unload(tmp);
  free(data);
  free(dst.pData);
  /* end first ftc */

  /* Mipmaps */
  j = 2;
  for (i = 1; i < mipmaps+1 ; i++, j = j * 2)
  {
      mw = w / j;
      mh = h / j;
      size = mw * mh * 3;
      fwrite(&size, sizeof(size), 1, u_fh);
      if (!quiet)
        printf("Uncompressed mipmap %d: %dx%d %d, size: %d\n", i, mw, mh, j, size);

      /* XXX allow chosing rescale type from the command line XXX */
      /* triangle filter (?)*/
      mip = FreeImage_Rescale(dib, mw, mh, filter);
      if (mip == NULL)
      {
          printf("Failed to rescale mipmap %d\n", i);
          return 0;
      }

      sw = FreeImage_GetPitch(mip);
      if (sw == 0)
      {
          printf("wtf? sw should not be 0!\n");
          return 0;
      }

      /* begin .ftu*/
      data = (BYTE*)malloc(mh * sw);
      FreeImage_ConvertToRawBits(data, mip, sw, 24, FI_RGBA_RED_MASK, FI_RGBA_GREEN_MASK, FI_RGBA_BLUE_MASK, FALSE);

      /* BGR again, should really convert the raw data */
      for (i2 = 0 ; i2 < mw * mh * 3 ; i2 += 3)
      {
          t = data[i2];
          data[i2] = data[i2 + 2];
          data[i2 + 2] = t;
      }

      fwrite(data, size, 1, u_fh);
      free(data);
      /* end .ftu */
      /* begin .ftc */

      tmp = FreeImage_ConvertTo32Bits(mip);
      if (tmp == NULL)
      {
        errx(EXIT_FAILURE, "%s: error while trying to convert mipmap %d to 32bits",
              progname, i);
      }
      sw = FreeImage_GetPitch(tmp);
      data = malloc(mh * sw);

      FreeImage_ConvertToRawBits(data, tmp, sw, 32, FI_RGBA_RED_MASK, FI_RGBA_GREEN_MASK, FI_RGBA_BLUE_MASK, FALSE);

      src.dwSize = sizeof(src);
      src.dwWidth = mw;
      src.dwHeight = mh;
      src.dwPitch = sw;
      src.format = ATI_TC_FORMAT_ARGB_8888;
      src.dwDataSize = mh * sw;
      src.pData = data;

      dst.dwSize = sizeof(dst);
      dst.dwWidth = mw;
      dst.dwHeight = mh;
      dst.dwPitch = 0;
      dst.format = ATI_TC_FORMAT_DXT1;
      dst.dwDataSize = ATI_TC_CalculateBufferSize(&dst);
      dst.pData = (ATI_TC_BYTE*) malloc(dst.dwDataSize);

      if (!quiet)
        printf("Compressing mipmap %d (%d : %d)\n", i, dst.dwDataSize, ((mw/4) * (mh/4)) * 8);

//      printf("Raw:\n");
//      pretty_print_block(src.pData, src.dwDataSize > 32 ? 32 : src.dwDataSize);

      ati_err = ATI_TC_ConvertTexture(&src, &dst, NULL, feedback, &i, NULL);
      if (!quiet)
        printf("\n");

      if (ati_err != ATI_TC_OK)
      {
        errx(EXIT_FAILURE, "%s: error %d while trying to compress texture.",
              progname, ati_err);
      }
//      printf("Compressed:\n");
//      pretty_print_block(dst.pData, dst.dwDataSize > 32 ? 32 : dst.dwDataSize);

      size = dst.dwDataSize;
      fwrite(&size, sizeof(size), 1, c_fh);
      fwrite(dst.pData, size, 1, c_fh);

      FreeImage_Unload(tmp);
      free(data);
      free(dst.pData);
      /* end .ftc */
      FreeImage_Unload(mip);
  }

  fclose(u_fh);
  fclose(c_fh);
  FreeImage_Unload(dib);
  return 0;
}
